# 機能設計書 44-Git FETCH_HEAD取得

## 概要

本ドキュメントは、cookbook（Redox OSのパッケージビルドシステム）におけるGit FETCH_HEAD取得機能の設計を記載する。この機能は、git fetchコマンド実行後のリモートブランチのコミットハッシュを取得する機能を提供する。

### 本機能の処理概要

Git FETCH_HEAD取得機能は、Gitリポジトリでフェッチ操作を行った後、.git/FETCH_HEADファイルから特定のリモートブランチに対応するコミットハッシュを取得する。これにより、ローカルのHEADとリモートの最新状態を比較し、更新の必要性を判定できる。

**業務上の目的・背景**：パッケージビルドシステムでは、ソースコードの更新があったかどうかを検出する必要がある。git fetchを実行した後、FETCH_HEADから取得したコミットハッシュとローカルのHEADを比較することで、再ビルドが必要かどうかを判定できる。これにより、不要なリビルドを回避し、ビルド時間を短縮できる。

**機能の利用シーン**：
- git fetch後のリモート状態確認
- ローカルとリモートのコミット比較
- ソース更新の必要性判定（can_skip_rebuild）
- 再フェッチ・再ビルドの判定

**主要な処理内容**：
1. .git/FETCH_HEADファイルを読み込む
2. 指定されたリモートURLとブランチ名に対応する行を検索する
3. マッチした行からコミットハッシュを抽出する
4. "not-for-merge"マーカーが付いた行は除外する

**関連システム・外部連携**：
- Gitリポジトリ（.git/FETCH_HEADファイル）

**権限による制御**：特になし（ファイル読み取り権限のみ必要）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面に直接関連せず、内部ユーティリティとして使用 |

## 機能種別

ユーティリティ / Git操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| dir | &PathBuf | Yes | Gitリポジトリのルートディレクトリパス | .git/FETCH_HEADファイルが存在すること |
| remote_url | &str | Yes | リモートリポジトリのURL | 空でないこと |
| remote_branch | &str | Yes | リモートブランチ名 | 空でないこと |

### 入力データソース

- ローカルファイルシステム上のGitリポジトリ
  - .git/FETCH_HEAD

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| コミットハッシュ | String | 40文字のSHA-1ハッシュ（16進数） |

### 出力先

- 関数の戻り値としてResult<String, String>で返却

## 処理フロー

### 処理シーケンス

```
1. get_git_fetch_rev関数呼び出し
   └─ ディレクトリパス、リモートURL、ブランチ名を受け取る
2. .git/FETCH_HEADファイルの読み込み
   └─ fs::read_to_string()
3. 検索パターンの構築
   └─ "branch '{branch}' of {url}"形式
4. 各行のスキャン
   └─ expected_comment_partを含む行を検索
5. "not-for-merge"チェック
   └─ このマーカーがある行は除外
6. コミットハッシュの抽出
   └─ 行の先頭の空白区切りでSHAを取得
7. 結果返却
   └─ マッチしなければエラー
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[.git/FETCH_HEAD読み込み]
    B --> C[検索パターン構築]
    C --> D[各行をスキャン]
    D --> E{パターンにマッチ?}
    E -->|No| F{次の行あり?}
    F -->|Yes| D
    F -->|No| G[エラー: 該当行なし]
    E -->|Yes| H{not-for-mergeを含む?}
    H -->|Yes| F
    H -->|No| I[行からSHAを抽出]
    I --> J[コミットハッシュを返却]
    G --> K[終了]
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-44-01 | マッチング形式 | "branch '{branch}' of {url}"形式で検索 | 常時 |
| BR-44-02 | not-for-merge除外 | "not-for-merge"を含む行はスキップ | マッチした行にマーカーがある場合 |
| BR-44-03 | 先頭SHA抽出 | 空白区切りの最初の要素がコミットハッシュ | マッチした行から抽出時 |
| BR-44-04 | 一致なしエラー | 該当する行が見つからない場合はエラー | 全行スキャン後 |

### 計算ロジック

```rust
// FETCH_HEAD検索のロジック
let expected_comment_part = format!("branch '{}' of {}", remote_branch, remote_url);

for line in fetch_head_content.lines() {
    if line.contains(&expected_comment_part) && !line.contains("not-for-merge") {
        let sha = line
            .split_whitespace()
            .next()  // 先頭の空白区切り要素がSHA
            .ok_or_else(|| "FETCH_HEAD line is malformed.".to_string())?;
        return Ok(sha.to_string());
    }
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| E-44-01 | ファイル不在 | .git/FETCH_HEADファイルが存在しない | エラーメッセージを返却 |
| E-44-02 | 読み取りエラー | ファイル読み取りに失敗 | エラーメッセージを返却 |
| E-44-03 | 該当行なし | 指定されたブランチ・URLの行が見つからない | エラーメッセージを返却 |
| E-44-04 | 行形式不正 | SHA抽出に失敗 | エラーメッセージを返却 |

### リトライ仕様

リトライは行わない。git fetch未実行の場合はFETCH_HEADが古いか存在しない可能性がある。

## トランザクション仕様

該当なし

## パフォーマンス要件

- FETCH_HEADは通常小さいファイルのため、線形スキャンで問題なし
- git fetch直後に呼び出されることを想定

## セキュリティ考慮事項

- ローカルファイルシステムのみにアクセス
- git fetch後のFETCH_HEADファイルの整合性に依存

## 備考

- FETCH_HEADのフォーマット例：
  ```
  a1b2c3d4e5f6... branch 'main' of https://github.com/user/repo
  d4e5f6a1b2c3... not-for-merge branch 'feature' of https://github.com/user/repo
  ```
- "not-for-merge"は現在チェックアウトしていないブランチに付与される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: FETCH_HEADのフォーマットを理解する

FETCH_HEADファイルのフォーマット理解が前提となる。

| フィールド | 説明 | 例 |
|-----------|------|-----|
| SHA | コミットハッシュ | a1b2c3d4e5f6... |
| マーカー | オプション | not-for-merge |
| ブランチ情報 | ブランチ名とURL | branch 'main' of https://... |

**読解のコツ**: FETCH_HEAD行のフォーマット
- `{SHA}\t\t{marker?}\tbranch '{branch}' of {url}`
- not-for-mergeマーカーはマージ対象でないブランチに付与

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | fs.rs | `src/cook/fs.rs` | get_git_fetch_rev関数（336-367行目）- メイン処理 |

**主要処理フロー**:
1. **341行目**: .git/FETCH_HEADパスの構築
2. **343-348行目**: FETCH_HEADファイルの読み込み
3. **350行目**: 検索パターンの構築（"branch '{branch}' of {url}"）
4. **352行目**: 各行のスキャン開始
5. **353行目**: パターンマッチングとnot-for-mergeチェック
6. **354-358行目**: コミットハッシュの抽出
7. **360行目**: 結果返却
8. **363-366行目**: マッチしなかった場合のエラー返却

#### Step 3: 呼び出し元を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | fetch.rs | `src/cook/fetch.rs` | fetch関数内のcan_skip_rebuild判定（240-247行目） |

**主要処理フロー**:
- **240行目**: get_git_fetch_revの呼び出し
- **241行目**: fetch_rev == head_revでスキップ可否を判定
- **242-244行目**: エラー時はログ出力してfalseを返却

### プログラム呼び出し階層図

```
fetch.rs
    │
    └─ fetch()
           │
           ├─ get_git_head_rev() → ローカルHEAD取得
           │
           ├─ get_git_remote_tracking() → リモート情報取得
           │
           └─ get_git_fetch_rev(dir, remote_url, remote_branch)
                  ├─ fs::read_to_string(".git/FETCH_HEAD")
                  ├─ lines().filter() → パターンマッチ
                  └─ split_whitespace().next() → SHA抽出
```

### データフロー図

```
[入力]                      [処理]                           [出力]

ディレクトリパス ────────▶ .git/FETCH_HEAD読み込み
リモートURL ─────────────▶                │
リモートブランチ ────────▶                │
                                          ▼
                          "branch '{branch}' of {url}"
                          パターン構築
                                          │
                                          ▼
                          各行スキャン
                                          │
                          ┌───────────────┴───────────────┐
                          ▼                               ▼
                    [マッチ]                        [マッチなし]
                          │                               │
                          ▼                               ▼
                    not-for-merge                    エラー返却
                    チェック
                          │
                          ▼
                    SHA抽出
                          │
                          ▼
                    String コミットハッシュ
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| fs.rs | `src/cook/fs.rs` | ソース | get_git_fetch_rev関数の実装 |
| fetch.rs | `src/cook/fetch.rs` | ソース | can_skip_rebuild判定での呼び出し元 |
| .git/FETCH_HEAD | ソースリポジトリ | Git内部 | git fetch後のリモート状態 |
